/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


#include	"StdAfx.h"


/*	This module contains the routines to search for arrangements
	of pieces that form the goal.  The algorithm is very primitive,
	as one can tell from the length of the module.  No attempt to
	optimize it has been made.  Some approaches for future versions
	are noted in Introduction.html.
*/


// Define a structure to describe the problem the Solver works on.
typedef struct
{
		Pieces			*pieces;	// pieces to try to fit into goal
		ShapeList		*work,		// goal to make and place to list solutions
						*last;		// last solution found so far
		HWND			window;		// window to receive Solver's messages
		WORD			status;		// status to be returned
		volatile Bool	quit;		// command to quit
}	Problem;
/*	The quit member is used to determine when the Solver should
	terminate before finding all solutions.  It is volatile
	because it is read and written by different threads.

	To prevent multithreading issues, the Solver thread is
	permitted only to read this variable.
*/


/*	Determine whether a piece fits into a goal.  If it does, place it.
	x and y give the translation to where the piece is to be fit.
	(This is a different fit routine from the one in Shapes.cpp.
	Here, we are trying to place pieces to fill the goal shape and
	consider them to fit as long as there is space, even if they
	are adjacent to other pieces.)
 */
static inline int	fits(Piece *piece, Shape *goal, int x, int y)
{
	int	i;


	/*	Test bounding rectangle.  If the requested translation of
		the piece would put its bounding rectangle outside the bounding
		rectangle of the goal, it won't fit.  We don't need to check
		whether x is less than zero due to the sort order of the
		points within a piece -- the first point always has x
		coordinate zero, so it will not be translated outside the goal.
	*/
	if (goal->width < x + piece->width)
		return 0;
	if (y < 0 || goal->height < y + piece->height)
		return 0;

	// See if there is a goal space everywhere the piece has a triangle.
	for (i = 0; i < piece->numPoints; i++)
		if (SPOT(goal, x + piece->points[i].x, y + piece->points[i].y) != GOAL)
			return 0;

	// Copy triangles from piece into goal.
	for (i = 0; i < piece->numPoints; i++)
		SPOT(goal, x + piece->points[i].x, y + piece->points[i].y) =
			piece->ID;

	return 1;
}


/*	Remove piece from goal.
	x and y give the translation to where the piece is in the goal.
*/
static inline void	removePiece(Piece *piece, Shape *goal, int x, int y)
{
	int	i;


	// Convert the spaces where the piece was back to goal spaces.
	for (i = 0; i < piece->numPoints; i++)
		SPOT(goal, x + piece->points[i].x, y + piece->points[i].y) = GOAL;
}


/*	Attempt to fill in a goal with pieces from a list.
	x and y give the coordinates of the first point
	not known not to be a goal space.
*/
static void	fill(Problem *problem, int x, int y)
{
	Pieces			* const pieces = problem->pieces;
	Shape			* const goal = problem->work->shape;

	PieceList	*q;
	Pieces		*p;
	int			yt;


	// First, find the lowest of the leftmost goal spaces in the goal.
	for (; x < goal->width; x++, y = 0)
	{
		for (; y < goal->height; y++)
			if (SPOT(goal, x, y) == GOAL)
				break;
		if (y < goal->height)
			break;
	}

	/*	If there is no empty space, goal is full.  That
		means we found a solution.  Add it to the list
		and notify the problem poser.
	*/
	if (goal->width <= x)
	{
		ShapeList	*newNode;

		
		// Append solution to list.
		newNode = (ShapeList *) malloc(sizeof *newNode);
		if (newNode == NULL)
		{
			DisplayCError("TriSolve: Solver");
			problem->status = SOLVER_ERROR;
			return;
		}
		newNode->next = NULL;
		newNode->shape = copyShape(problem->work->shape);
		if (newNode->shape == NULL)
		{
			problem->status = SOLVER_ERROR;
			free(newNode);
			return;
		}
		problem->last = problem->last->next = newNode;

		// Inform the problem poser there is a new solution.
		PostMessage(problem->window, WM_SOLUTION_FOUND, 0, 0);
		return;
	}

	// Now try putting each piece in the space we found in the goal.
	for (p = pieces; p != NULL; p = p->next)
	{
		// If piece is already used, skip it.
		if (p->used)
			continue;

		// Try each orientation of the piece.
		for (q = p->list; q != NULL; q = q->next)
		{
			/*	Find the translation that moves the lowest of
				the leftmost triangles in the piece to the lowest
				of the leftmost spaces in the goal.  The
				horizontal component is always x since the first
				point in a normalized piece always has x coordinate
				zero.  The vertical component is y minus the y
				coordinate of the first point.
			*/
			yt = y - q->piece->points[0].y;

			// If triangle is pointed in wrong direction, go on.
			if (x+yt+goal->parity+q->piece->parity & 1)
				continue;

			// So far, so good.  Does the piece fit in the goal?
			if (fits(q->piece, goal, x, yt))
			{
				p->used = TRUE;
				fill(problem, x, y+1);
				p->used = FALSE;
				removePiece(q->piece, goal, x, yt);
				if (problem->quit || problem->status != SOLVER_SUCCESS)
					return;
			}
		}
	}
}


/*	This is the main routine for the thread.  Primarily, it
	dispatches to the search engine, the fill routine.
*/
static void	SolverMain(Problem *problem)
{
	/*	The search for solutions is compute-bound, so lower the
		thread priority to avoid taking control away from the user
		and general applications.
	*/
	SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_BELOW_NORMAL);

	// Try to solve the problem.
	if (problem->work == NULL)
		problem->status = SOLVER_ERROR;
	else
	{
		// Mark shape volatile while we work on it.
		problem->work->shape->isVolatile = TRUE;
		fill(problem, 0, 0);
		problem->work->shape->isVolatile = FALSE;
	}

	if (problem->quit)
		problem->status = SOLVER_ABORTED;

	// Tell poser we are done.
	PostMessage(problem->window, WM_SOLVER_ENDING, problem->status, 0);
}


// This routine can be called to stop the Solver thread prematurely.
void	StopSolver(void *problem)
{
	/*	The Solver checks the quit member frequently, so we
		can get it to quit by setting the flag.
	*/
	((Problem *)problem)->quit = TRUE;

	/*	There doesn't seem to be any C run-time library to terminate
		a thread externally, so we have to trust the thread to complete.

		Alternatively, we could ensure the thread uses no C run-time
		library routines and convert the thread calls to Windows'
		CreateThread and associated calls.  If so, be sure to close
		the handle with CloseHandle.  Currently, we call copyShape,
		and it uses malloc.  We would also want some way to clean up
		memory (or other resources) the thread had allocated.
	*/
}


#include	<process.h>


/*	This routine is called to start a thread that searches for a
	solution.  It is passed data that specifies a problem to
	solve and a handle to a window to be notified about Solver
	status.

	Results are passed back in a list which only the Solver
	thread is allowed to write to until it completes, and the Solver
	must guarantee the list is always in a consistent state.
*/
void *StartSolver(HWND window, Pieces *pieces, ShapeList *work)
{
	static const char	title[] = "TriSolve: Starting Solver thread";

	Problem			*problem;


	problem = (Problem *) malloc(sizeof *problem);
	if (problem == NULL)
	{
		DisplayCError(title);
		return NULL;
	}

	problem->window	= window;
	problem->pieces	= pieces;
	problem->work	=
	problem->last	= work;
	problem->status	= SOLVER_SUCCESS;
	problem->quit	= FALSE;

	if (NO_THREAD == _beginthread((void (*)(void *)) SolverMain, 0, problem))
	{
		DisplayWindowsError(title);
		free(problem);
		return NULL;
	}

	return problem;
}