/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


#include	"StdAfx.h"
#include	"resource.h"


#define	NUMDIGITS	9	// number of digits allowed in edit box
#define	BUFLEN		80	// temporary message buffer, for string formatting


// Turn the solution navigation controls on or off.
static void	EnableGoto(HWND dialog, Bool flag)
{
	EnableWindow(GetDlgItem(dialog, IDC_VIEWGROUP       ),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_FIRSTSOLUTION   ),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_PREVIOUSSOLUTION),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_GOTOSOLUTION    ),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_NEXTSOLUTION    ),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_LASTSOLUTION    ),	flag);
}


// Turn the status dialog box controls on or off.
static void	EnableStatus(HWND dialog, Bool flag)
{
	EnableWindow(GetDlgItem(dialog, IDC_SLOW  ),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_MEDIUM),	flag);
	EnableWindow(GetDlgItem(dialog, IDC_FAST  ),	flag);

	/*	The solution navigation controls are enabled when we are
		not automatically updating the solution display -- which
		is when the other controls are disabled (Solver is not
		running) or we are in fast mode (no automatic display).
	*/
	EnableGoto(dialog, !flag || IsDlgButtonChecked(dialog, IDC_FAST));
}


// Set the solution number edit box as desired.  Show "<work>" for zero.
static void	setSolutionNumber(HWND dialog, int numCurrent)
{
	if (numCurrent == 0)
		SetDlgItemText(dialog, IDC_GOTOSOLUTION, "<work>");
	else
		SetDlgItemInt(dialog, IDC_GOTOSOLUTION, numCurrent, FALSE);
}


/*	Move current pointer to a specified solution.

	zero points to the list header (the zeroth solution).
	numCurrent and current are updated with the new
	solution number and a pointer to it.
*/
static void	moveToSolution(HWND dialog, ShapeList *zero,
	int newNum, int *numCurrent, ShapeList **current, WORD ID)
{
	/*	We have only a forward-linked list of solutions.  If the
		desired solution is after the current solution, we can
		start following the list from the current spot.  Otherwise,
		we have to start at the beginning.  In either case, we
		then iterate through the list links one-by-one.  This has
		proven to be fast even for hundreds of thousands of
		solutions.  If memory is tight, though, this could cause
		immense numbers of page swaps.  In such case, we could
		improve performance by building a structure to provide
		O(log n) time navigation of the solutions.  This should be
		done when receiving the WM_SOLUTION_FOUND message, below,
		using memory entirely owned by this thread and not involving
		the Solver thread.  That is, the navigation structure would
		be entirely external to the list built by the Solver -- it
		would be our index into that list.
	*/

	// If moving to earlier position, start at beginning.
	if (newNum < *numCurrent)
	{
		*numCurrent = 0;
		*current = zero;
	}

	//	Follow list links until desired solution is reached.
	for (; *numCurrent < newNum && *current != NULL; ++*numCurrent)
		*current = (*current)->next;

	// If we go beyond the end, show the goal/work shape.
	if (*current == NULL)
	{
		*current = zero;
		*numCurrent = 0;
	}

	// Tell the parent we have changed what we are displaying.
	SendMessage(GetParent(dialog), WM_SETSHAPE,
		MAKELONG(ID, 0), (long) (*current)->shape);
}


/*	Message handler for status box.

	On initialization, the dialog must be passed a pointer to a StatusInfo
	structure with the ID filled in.  The work member need not be initialized
	(since it will be used until the Solver is started), but it is shared by
	this dialog and other code and threads.  (Yuck.  Clean it up before
	using further.)

	Before starting the Solver or changing the work structure, this routine
	must be sent WM_SOLVER_STARTING.  It frees old data in the work
	structure, if any, and initializes its private data.  Then the Solver
	may be started and may write to the work structure, filling in solutions.
	Threads other than the Solver may read the work structure.
*/
BOOL CALLBACK	Status(HWND dialog, UINT message, WPARAM wParam, LPARAM lParam)
{
	/*	On creation, we are passed a pointer to space allocated for us.
		On subsequent calls, we use our saved copy of that pointer.
	*/
	StatusInfo	* const info = (StatusInfo *) (message == WM_INITDIALOG
		? lParam : GetWindowLong(dialog, GWL_USERDATA));

	char		buffer[BUFLEN];


	switch (message)
	{
		case WM_INITDIALOG:
			// Save pointer passed to us at creation for future calls.
			SetWindowLong(dialog, GWL_USERDATA, (LONG) info);

			// Initialize data.
			info->work.next		= NULL;
			info->work.shape	= NULL;
			info->current		= &info->work;
			info->numCurrent	= 0;
			info->numSolutions	= 0;

			// Initialize controls.
			SendDlgItemMessage(dialog, IDC_GOTOSOLUTION, EM_SETLIMITTEXT, NUMDIGITS, 0);
			SendDlgItemMessage(dialog, IDC_GOTOSOLUTION, EM_LIMITTEXT, NUMDIGITS, 0);
			SetDlgItemInt(dialog, IDC_GOTOSOLUTION, 1, FALSE);
			CheckRadioButton(dialog, IDC_FAST, IDC_SLOW, IDC_MEDIUM);

			// Get bitmaps to display on-off switch in dialog box.
			info->solverOn = (HBITMAP) LoadImage(
				(HINSTANCE) GetWindowLong(dialog, GWL_HINSTANCE),
				MAKEINTRESOURCE(IDB_SOLVERON),
				IMAGE_BITMAP, 0, 0, LR_LOADMAP3DCOLORS);
			info->solverOff = (HBITMAP) LoadImage(
				(HINSTANCE) GetWindowLong(dialog, GWL_HINSTANCE),
				MAKEINTRESOURCE(IDB_SOLVEROFF),
				IMAGE_BITMAP, 0, 0, LR_LOADMAP3DCOLORS);
			SendMessage(GetDlgItem(dialog, IDC_SWITCH), STM_SETIMAGE,
				IMAGE_BITMAP, (LPARAM) info->solverOff);

			return TRUE;

		/*	When the system colors change, we have to reload the bitmaps
			to replace their backgrounds with the new color.
		*/
		case WM_SYSCOLORCHANGE:
			DeleteObject(info->solverOn);
			DeleteObject(info->solverOff);
			info->solverOn = (HBITMAP) LoadImage(
				(HINSTANCE) GetWindowLong(dialog, GWL_HINSTANCE),
				MAKEINTRESOURCE(IDB_SOLVERON),
				IMAGE_BITMAP, 0, 0, LR_LOADMAP3DCOLORS);
			info->solverOff = (HBITMAP) LoadImage(
				(HINSTANCE) GetWindowLong(dialog, GWL_HINSTANCE),
				MAKEINTRESOURCE(IDB_SOLVEROFF),
				IMAGE_BITMAP, 0, 0, LR_LOADMAP3DCOLORS);
			SendMessage(GetDlgItem(dialog, IDC_SWITCH), STM_SETIMAGE,
				IMAGE_BITMAP,
				IsWindowEnabled(GetDlgItem(dialog, IDC_SLOW)) ?
					(LPARAM) info->solverOn : (LPARAM) info->solverOff);
			return TRUE;

		case WM_MOVING:
		case WM_SIZING:
			// Disable moving and sizing by setting new location to current location.
			GetWindowRect(dialog, (RECT *) lParam);
			return TRUE;

		case WM_LBUTTONDOWN:
			/*	When the user clicks in the dialog box, we want to take
				the keyboard focus.  Here we get a button-down message
				and set the focus.  We also tell the parent window to
				display our shape.

				Some button-down messages are absorbed by or bypass the
				dialog box window procedure and sent to child windows.
				Then it is up to the child window to take the focus.
			*/
			SendMessage(GetParent(dialog), WM_SETSHAPE,
				MAKELONG(info->ID, CE_IMPERATIVE), (long) info->current->shape);
			SetFocus(dialog);
			return FALSE;

		case WM_COMMAND:
			/*	When the user has been interacting with our controls, change
				the exhibit to show our display.  Note that edit box changes
				while the edit box is disabled are due to solution updates, not
				user action, so this does not count as user interaction.  We can
				catch user interaction with the edit box by catching it setting
				focus.  For other controls, any WM_COMMAND represents some
				user interaction.
			*/
			if (	LOWORD(wParam) != IDC_GOTOSOLUTION ||
					HIWORD(wParam) == EN_SETFOCUS
			)
				SendMessage(GetParent(dialog), WM_SETSHAPE,
					MAKELONG(info->ID, CE_IMPERATIVE), (long) info->current->shape);

			switch (LOWORD(wParam))
			{
				// Treat click on switch as Start/Stop Solver command.
				case IDC_SWITCH:
					switch (HIWORD(wParam))
					{
						case STN_CLICKED:
							// User clicked the Solver switch bitmap.
							SendMessage(GetParent(dialog), WM_COMMAND,
								MAKELONG(IDM_START_SOLVING, 0), (LPARAM) dialog);
							return TRUE;
					}
					break;

				case IDC_FAST:
					// Kill timer in case leaving slow mode.
					KillTimer(dialog, 1);
					/*	Show the most recent solution.  If we leave "<work>"
						in the edit box, the user will have to remove it.
					*/
					if (0 < info->numSolutions)
						setSolutionNumber(dialog, info->numSolutions);
					// Allow the user to move through solutions so far.
					EnableGoto(dialog, TRUE);
					return TRUE;

				case IDC_MEDIUM:
					// Kill timer in case leaving slow mode.
					KillTimer(dialog, 1);
					// Disable controls for moving through solutions.
					EnableGoto(dialog, FALSE);
					return TRUE;

				case IDC_SLOW:
					// Display work in progress.
					setSolutionNumber(dialog, 0);
					// Disable controls for moving through solutions.
					EnableGoto(dialog, FALSE);
					// Start a timer to display the work in progress periodically.
					if (SetTimer(dialog, 1, 100, NULL) == 0)
						DisplayWindowsError("TriSolve: Setting timer");
					return TRUE;

				/*	When controls for moving through solutions are clicked,
					move appropriately.
				*/
				case IDC_FIRSTSOLUTION:
					// Go to first solution if there are any.
					if (0 < info->numSolutions)
						setSolutionNumber(dialog, 1);
					return TRUE;
				case IDC_PREVIOUSSOLUTION:
					// Go to previous solution, if there are any, and stop at 1.
					if (0 < info->numSolutions)
						setSolutionNumber(dialog,
							info->numCurrent <= 1 ? 1 : info->numCurrent-1);
					return TRUE;
				case IDC_NEXTSOLUTION:
					// Go to next solution, if there are any, and stop at last.
					if (0 < info->numSolutions)
						setSolutionNumber(dialog,
							info->numCurrent < info->numSolutions ?
								info->numCurrent+1 : info->numSolutions);
					return TRUE;
				case IDC_LASTSOLUTION:
					// Go to last solution, if there are any.
					if (0 < info->numSolutions)
						setSolutionNumber(dialog, info->numSolutions);
					return TRUE;

				case IDC_GOTOSOLUTION:
					switch (HIWORD(wParam))
					{
						// Edit control changed.  Display the newly selected shape.
						case EN_CHANGE:
							moveToSolution(dialog, &info->work,
								GetDlgItemInt(dialog, IDC_GOTOSOLUTION, NULL, FALSE),
								&info->numCurrent, &info->current, info->ID);
							return TRUE;
					}
					return FALSE;
			}
			break;
			
		case WM_GETSHAPE:
			// Tell caller what shape we are displaying.
			* (Shape **) lParam = info->current->shape;
			return TRUE;
			
		case WM_SOLVER_STARTING:
			/*	The Solver is starting.  Free the old work, if any.
				This could be done by the sender, but it is a little
				reassuring to know that we are here and not in any other
				code using the shape when we free it.  Of course, we
				would not be in any other code anyway, since the sender
				of this message is in the same thread as we are now, so
				the sender could free it just as well.

				And of course, there must be no Solver (in another
				thread) using this data -- it is not allowed to start
				a Solver on a work area while another Solver is still
				using it.
			*/
			freeShapeList(info->work.next);
			freeShape(info->work.shape);
			info->work.next = NULL;

			/*	Record the new shape and initialize our knowledge of
				how many solutions have been found and our position in them.
			*/
			info->work.shape = (Shape *) lParam;
			info->numSolutions = 0;
			info->numCurrent = 0;
			info->current = &info->work;
			SetDlgItemText(dialog, IDC_STATUS,
				"Searching for solutions. None found so far...");
			setSolutionNumber(dialog, 0);

			// In slow mode, start the timer (see IDC_SLOW and WM_TIMER cases).
			if (IsDlgButtonChecked(dialog, IDC_SLOW))
				if (SetTimer(dialog, 1, 100, NULL) == 0)
					DisplayWindowsError("TriSolve: Setting timer");

			// Enable the speed selection while the Solver runs.
			EnableStatus(dialog, TRUE);

			// Turn on switch.
			SendMessage(GetDlgItem(dialog, IDC_SWITCH), STM_SETIMAGE,
				IMAGE_BITMAP, (LPARAM) info->solverOn);
			return TRUE;

		case WM_SOLUTION_FOUND:
			// Update status message.
			wsprintf(buffer,
				"Searching for solutions. %d found so far...",
				++info->numSolutions);
			SetDlgItemText(dialog, IDC_STATUS, buffer);
			// At medium speed, track solutions in edit box.
			if (IsDlgButtonChecked(dialog, IDC_MEDIUM))
				setSolutionNumber(dialog, info->numSolutions);
			return TRUE;

		case WM_SOLVER_ENDING:
			// Stop updating the work in progress.
			KillTimer(dialog, 1);

			// Report results to user.
			if (wParam != SOLVER_ABORTED)
				MessageBeep(MB_OK);
			if (wParam == SOLVER_ABORTED)
				SetDlgItemText(dialog, IDC_STATUS, "Solver aborted.");
			else if (wParam == SOLVER_ERROR)
				SetDlgItemText(dialog, IDC_STATUS, "Solver ended with error.");
			else if (info->numSolutions == 0)
				SetDlgItemText(dialog, IDC_STATUS,
					"Done. There are no solutions.");
			else if (info->numSolutions == 1)
				SetDlgItemText(dialog, IDC_STATUS,
					"Done. There is only one solution.");
			else
			{
				wsprintf(buffer, "Done. There are %d solutions.", info->numSolutions);
				SetDlgItemText(dialog, IDC_STATUS, buffer);
			}

			// Disable speed controls and enable solution navigation controls.
			EnableStatus(dialog, FALSE);

			// Turn on switch.
			SendMessage(GetDlgItem(dialog, IDC_SWITCH), STM_SETIMAGE,
				IMAGE_BITMAP, (LPARAM) info->solverOff);

			if (info->numSolutions == 0)
				// If there are no solutions, display the goal shape.
				setSolutionNumber(dialog, 0);
			else
				/*	If there are solutions and none is selected (we are
					showing the work area), show the first solution.
				*/
				if (0 == GetDlgItemInt(dialog, IDC_GOTOSOLUTION, NULL, FALSE))
					setSolutionNumber(dialog, 1);
			return TRUE;

		case WM_SHOWMESSAGE:
			SetDlgItemText(dialog, IDC_STATUS, (LPCTSTR) lParam);
			return TRUE;

		case WM_TIMER:
			// When we get a timer message, update the display of work in progress.

			/*	First, check if we are still in the display-work mode.  If
				not, stop the timer.  If still in display-work mode, tell
				parent its time to update display.
			*/
			if (!IsDlgButtonChecked(dialog, IDC_SLOW))
				KillTimer(dialog, 1);
			else
				SendMessage(GetParent(dialog), WM_SETSHAPE,
					MAKELONG(info->ID, 0), (long) info->current->shape);
			return TRUE;

		case WM_DESTROY:
			DeleteObject(info->solverOn);
			DeleteObject(info->solverOff);
			return TRUE;
	}
	return FALSE;
}